import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.logging.Logger;

public class GDBakMain {
	
	public static final String APPNAME = "konskar-gdbak-1.0.0";
	private static final Logger logger = 
		Logger.getLogger(GDBakMain.class.getName());
	private final Properties props = new Properties();
	private final List<GDBak> list = new ArrayList<GDBak>(1);
	private String bakdir;

	public static void main(String[] args) {
		logger.info("Google Data backup start");
		GDBakMain gdbak = new GDBakMain();
		gdbak.loadProperties();
		gdbak.fillList();
		gdbak.connect(args.length >= 1 ? args[0] : null, 
				args.length == 2 ? args[1] : null);
		gdbak.backup();
		logger.info("Google Data backup complete");
	}

	private void backup() {
		for (GDBak gdbak : list) {
			try {
				gdbak.backup();
			} catch (Exception e) {
				logger.severe(e.getMessage());
				logger.config(getStackTrace(e));
			}
		}
	}

	private void connect(String email, String passwd) {
		do {
			if(email == null) {
				email = System.console().readLine(
						"Enter your email [your_email@gmail.com]: ");
			}
			if(passwd == null) {
				passwd = new String(System.console().readPassword(
						"Enter your password [your_password]: "));
			}
			try {
				for(GDBak gdbak : list) {
					gdbak.connect(email, passwd);
				}
				logger.info("Authentication succeeded");
				return;
			} catch (Exception e) {
				email = null;
				passwd = null;
				logger.severe(e.getMessage());
				logger.config(getStackTrace(e));
			}
		} while (true);
	}

	private String getStackTrace(Exception e) {
		StringBuffer buffer = new StringBuffer();
		buffer.append(e.getClass().getCanonicalName()).append(": ");
		buffer.append(e.getMessage()).append("\n");
		for (StackTraceElement ste : e.getStackTrace()) {
			buffer.append("\tat ").append(ste.getClassName()).append(".");
			buffer.append(ste.getMethodName()).append(" (");
			buffer.append(ste.getFileName()).append(":");
			buffer.append(ste.getLineNumber()).append(")\n");
		}
		return buffer.toString();
	}

	private void fillList() {
		if(Boolean.valueOf(getProperty("gdbak.mail")).booleanValue()) {
			list.add(new GDBakMail(props, getPath("gdbak.mail.dir")));
		}
		if(Boolean.valueOf(getProperty("gdbak.docs")).booleanValue()) {
			list.add(new GDBakDocs(props, getPath("gdbak.docs.dir")));
		}
		if(Boolean.valueOf(getProperty("gdbak.calendar")).booleanValue()) {
			list.add(new GDBakCalendar(
					getProperty("gdbak.calendar.url"), 
					getPath("gdbak.calendar.file")));
		}
		if(Boolean.valueOf(getProperty("gdbak.contacts")).booleanValue()) {
			list.add(new GDBakContacts(
					getProperty("gdbak.contacts.url"), 
					getPath("gdbak.contacts.file")));
		}
		if(Boolean.valueOf(getProperty("gdbak.reader")).booleanValue()) {
			list.add(new GDBakHttp(props, 
					getProperty("gdbak.reader.url"),
					getPath("gdbak.reader.file")));
		}
		if(Boolean.valueOf(getProperty("gdbak.bookmarks")).booleanValue()) {
			list.add(new GDBakHttp(props, 
					getProperty("gdbak.bookmarks.url"),
					getPath("gdbak.bookmarks.file")));
		}
		if(Boolean.valueOf(getProperty("gdbak.igoogle")).booleanValue()) {
			list.add(new GDBakHttp(props, 
					getProperty("gdbak.igoogle.url"),
					getPath("gdbak.igoogle.file")));
		}
		if(list.isEmpty()) {
			logger.info("No Backups have been selected");
			System.exit(0);
		}
	}

	private void loadProperties() {
		try {
			props.load(GDBakMain.class.getResourceAsStream("gdbak.properties"));
			props.putAll(System.getProperties());
			props.load(new FileInputStream(
					props.getProperty("gdbak.props", "gdbak.properties")));
		} catch (IOException e) {
			logger.info(e.getMessage());
		}
		bakdir = props.getProperty("gdbak.dir", props.getProperty("user.dir"));
	}

	private File getPath(String key) {
		return new File(bakdir, getProperty(key)).getAbsoluteFile();
	}

	private String getProperty(String key) {
		return props.getProperty(key);
	}

}
