import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Properties;
import java.util.logging.Logger;

import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Store;

public class GDBakMail implements GDBak {

	private static final Logger logger =
		Logger.getLogger(GDBakMail.class.getName());
	private Store store;
	private Properties props;
	private File parent;

	public GDBakMail(Properties props, File parent) {
		this.props = props;
		this.parent = parent;
	}

	@Override
	public void backup() throws MessagingException, IOException {
		parent.mkdirs();
		String label = props.getProperty("gdbak.mail.label");
		if(label.equalsIgnoreCase("all")) {
			for(Folder folder : store.getDefaultFolder().list()) {
				backup(folder);
			}
		}
		else {
			backup(store.getFolder(label));
		}
		store.close();
		logger.info("Google Mail backup complete");
	}

	private void backup(Folder folder) throws MessagingException, IOException {
		folder.open(Folder.READ_ONLY);
		backup(folder.getMessages());
		folder.close(true);
	}

	private void backup(Message[] messages) throws MessagingException, IOException {
		SimpleDateFormat sdf = new SimpleDateFormat(
				props.getProperty("gdbak.mail.name"));
		String ext = props.getProperty("gdbak.mail.ext");
		for (Message message : messages) {
			File file = new File(parent, sdf.format(message.getSentDate()) 
					+ "_" + message.getMessageNumber() + "." + ext);
			if(!file.exists()) {
				write(message, file);
			}
		}
	}

	private void write(Message message, File file) throws IOException, MessagingException {
		FileOutputStream fos = null;
		try {
			fos = new FileOutputStream(file);
			message.writeTo(fos);
			logger.config("write to " + file.toString());
		}finally {
			if(fos != null) {
				fos.flush();
				fos.close();
			}
		}
	}
	
	@Override
	public void connect(String email, String passwd) throws MessagingException {
		Session session = Session.getDefaultInstance(props);
		store = session.getStore(props.getProperty("mail.store.protocol"));
		store.connect(props.getProperty("gdbak.mail.host"), email, passwd);
	}

}
