import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.List;
import java.util.Properties;
import java.util.logging.Logger;

import com.google.gdata.client.GoogleService;
import com.google.gdata.client.GoogleAuthTokenFactory.UserToken;
import com.google.gdata.client.docs.DocsService;
import com.google.gdata.data.MediaContent;
import com.google.gdata.data.docs.DocumentListEntry;
import com.google.gdata.data.docs.DocumentListFeed;
import com.google.gdata.data.media.MediaSource;
import com.google.gdata.util.AuthenticationException;
import com.google.gdata.util.ServiceException;

public class GDBakDocs implements GDBak {

	private static final Logger logger =
		Logger.getLogger(GDBakDocs.class.getName());
	private DocsService docsService;
	private GoogleService googleService;
	private Properties props;
	private File parent;

	public GDBakDocs(Properties props, File parent) {
		this.docsService = new DocsService(GDBakMain.APPNAME);
		this.googleService = new GoogleService("wise", GDBakMain.APPNAME);
		this.props = props;
		this.parent = parent;
	}

	@Override
	public void backup() throws IOException, ServiceException {
		URL url = new URL(props.getProperty("gdbak.docs.url"));
		DocumentListFeed feed = docsService.getFeed(
				url, DocumentListFeed.class);
		backup(feed.getEntries());
		logger.info("Google Docs backup complete");
	}

	private void backup(List<DocumentListEntry> entries) throws IOException, ServiceException {
		parent.mkdirs();
		String docExt = props.getProperty("gdbak.document.ext");
		String preExt = props.getProperty("gdbak.presentation.ext");
		String draExt = props.getProperty("gdbak.drawing.ext");
		String sprExt = props.getProperty("gdbak.spreadsheet.ext");
		for (DocumentListEntry entry : entries) {
			if (entry.getType().equalsIgnoreCase("document")) {
				write(entry, docExt);
			}else if (entry.getType().equalsIgnoreCase("presentation")) {
				write(entry, preExt);
			}else if (entry.getType().equalsIgnoreCase("drawing")) {
				write(entry, draExt);
			}else if (entry.getType().equalsIgnoreCase("spreadsheet")) {
				UserToken docsToken = (UserToken) docsService
						.getAuthTokenFactory().getAuthToken();
				UserToken spreadsheetsToken = (UserToken) googleService
						.getAuthTokenFactory().getAuthToken();
				docsService.setUserToken(spreadsheetsToken.getValue());
				write(entry, sprExt);
				docsService.setUserToken(docsToken.getValue());			
			}
		}
	}

	private void write(DocumentListEntry entry, String ext) throws IOException, ServiceException {
		File file = new File(parent, entry.getTitle().getPlainText()+ "." + ext);
		if(!file.exists() || entry.getUpdated().getValue() > file.lastModified()) {
			String exportUrl = ((MediaContent) entry.getContent()).getUri()
					+ "&exportFormat=" + ext;
			MediaContent mc = new MediaContent();
			mc.setUri(exportUrl);
			MediaSource ms = docsService.getMedia(mc);
			write(ms.getInputStream(), file);
		}
	}

	private void write(InputStream inputStream, File file) throws IOException {
		BufferedInputStream bis = null;
		BufferedOutputStream bos = null;
		try {
			bis = new BufferedInputStream(inputStream);
			bos = new BufferedOutputStream(new FileOutputStream(file), 1024);
			int count;
			byte data[] = new byte[1024];
			while ((count = bis.read(data)) != -1) {
				bos.write(data, 0, count);
			}
			logger.config("write to " + file.toString());
		} finally {
			if (bis != null) {
				bis.close();
			}
			if (bos != null) {
				bos.flush();
				bos.close();
			}
		}
	}

	@Override
	public void connect(String email, String passwd) throws AuthenticationException {
		docsService.setUserCredentials(email, passwd);
		googleService.setUserCredentials(email, passwd);
	}

}
